# JavaScript Plugin Tips

## Which Plugin Type to Use?

| If you want to... | Use |
|-------------------|-----|
| Create a shape | `thirdPartyJavaScriptShape` |
| Modify an existing shape | `thirdPartyJavaScriptDeformer` |
| Generate a value (number, colour, string) | `thirdPartyJavaScript` |

---

## Common Patterns

### Expression-Based Output

JavaScript plugins use an expression-based model where the result of the last expression becomes the output value. There is no need for a `return` statement at the top level.

```javascript
Math.sin(time * frequency) * amplitude;
```

This is standard JavaScript behaviour - the same as how `eval()` and REPL environments work.

### Complex Logic with IIFEs

For scripts requiring control flow (conditionals, early returns), use an Immediately Invoked Function Expression (IIFE):

```javascript
(function() {
    if (time < 0) {
        return 0;
    }
    if (time > 100) {
        return amplitude;
    }
    return Math.sin(time * frequency) * amplitude;
})();
```

### Variable Names

Input attributes are available as JavaScript variables using their exact attribute names from `definitions.json`. For example:

```json
"attributes": {
    "time": { "type": "double" },
    "frequency": { "type": "double" },
    "amplitude": { "type": "double" }
}
```

These become the variables `time`, `frequency`, and `amplitude` in your script.

---

## Supported Input Types

| Type | JavaScript Representation |
|------|---------------------------|
| `double` | number |
| `int` | number |
| `bool` | boolean |
| `string` | string |
| `double2` | `{ x, y }` |
| `double3` | `{ x, y, z }` |
| `int2` | `{ x, y }` |
| `color` | `{ r, g, b, a }` (0-255 range) |

---

## Utility Plugins (thirdPartyJavaScript)

Utility plugins generate output values. The result of the last expression becomes the output.

### Simple Expression

```javascript
Math.sin(time * frequency * Math.PI * 2) * amplitude;
```

### Returning Complex Types

To return a `double2`, `double3`, `int2`, or `color`, return an object with the appropriate properties:

```javascript
// Returns a double2
({ x: Math.cos(time) * radius, y: Math.sin(time) * radius });
```

```javascript
// Returns a color
({ r: 255, g: Math.floor(time) % 256, b: 128, a: 255 });
```

Note the parentheses around the object literal - this prevents JavaScript from interpreting the braces as a code block.

### Random Values

Use `cavalry.random()` for deterministic random values:

```javascript
// cavalry.random(min, max, seed, offset)
var randomX = cavalry.random(-100, 100, seed, 0);
var randomY = cavalry.random(-100, 100, seed, 1);
({ x: randomX, y: randomY });
```

---

## Shape Plugins (thirdPartyJavaScriptShape)

Shape plugins create geometry by returning a `cavalry.Mesh` or `cavalry.Path` object.

### Basic Shape

```javascript
(function() {
    var path = new cavalry.Path();
    path.addEllipse(0, 0, radius, radius);

    var mesh = new cavalry.Mesh();
    mesh.addPath(path);
    return mesh;
})();
```

### Custom Geometry with moveTo/lineTo

```javascript
(function() {
    var path = new cavalry.Path();

    // Draw a diamond
    path.moveTo(0, -radius);
    path.lineTo(radius, 0);
    path.lineTo(0, radius);
    path.lineTo(-radius, 0);
    path.close();

    var mesh = new cavalry.Mesh();
    mesh.addPath(path);
    return mesh;
})();
```

### Parametric Curves

```javascript
(function() {
    var path = new cavalry.Path();

    for (var i = 0; i <= segments; i++) {
        var theta = (i / segments) * Math.PI * 2;
        var r = innerRadius + lobeSize * Math.cos(lobes * theta);
        var x = r * Math.cos(theta);
        var y = r * Math.sin(theta);

        if (i === 0) {
            path.moveTo(x, y);
        } else {
            path.lineTo(x, y);
        }
    }
    path.close();

    var mesh = new cavalry.Mesh();
    mesh.addPath(path);
    return mesh;
})();
```

### Materials

When adding paths to a mesh, the material argument is optional:

```javascript
mesh.addPath(path);                        // No material - inherits from parent
mesh.addPath(path, new cavalry.Material()); // Explicit material in code
```

**No material:** The path inherits its appearance from the parent mesh. For shape plugins that extend `ShapeNode`, this means users can control fill and stroke via the standard Cavalry UI (Fill/Stroke tabs).

**Explicit material:** Use when you need programmatic control over colours or stroke settings, or when building complex meshes with different materials per path.

---

## Deformer Plugins (thirdPartyJavaScriptDeformer)

Deformer plugins modify existing shapes using the `def` module.

### Basic Point Manipulation

```javascript
function visitAllMeshes(mesh, func) {
    for (var i = 0; i < mesh.childMeshCount(); i++) {
        var child = mesh.getChildMeshAtIndex(i);
        visitAllMeshes(child, func);
        mesh.setChildMeshAtIndex(i, child);
    }
    func(mesh);
}

function deformMesh(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        var pd = path.pathData();

        for (var idx = 0; idx < pd.length; idx++) {
            var pt = pd[idx].point;
            var normal = pd[idx].normal;

            // Move point along its normal
            pd[idx].point.x += normal.x * amount;
            pd[idx].point.y += normal.y * amount;
        }

        path.setPathData(pd);
        mesh.setPathAtIndex(p, path);
    }
}

var root = def.getRootMesh();
visitAllMeshes(root, deformMesh);
def.setRootMesh(root);
```

### Using Falloffs

Falloffs allow localised effects. Use `def.getFalloffAtPoint(x, y)` to get the falloff value (0-1) at any position:

```javascript
function deformMesh(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        var pd = path.pathData();

        for (var idx = 0; idx < pd.length; idx++) {
            var pt = pd[idx].point;
            var normal = pd[idx].normal;

            // Get falloff at this point (includes Strength)
            var falloff = def.getFalloffAtPoint(pt.x, pt.y);
            if (falloff <= 0) continue;

            pd[idx].point.x += normal.x * amount * falloff;
            pd[idx].point.y += normal.y * amount * falloff;
        }

        path.setPathData(pd);
        mesh.setPathAtIndex(p, path);
    }
}
```

### Sampling Along Paths

Use path sampling methods to place objects or effects along path edges:

```javascript
function addEffectsAlongPath(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        if (!path || path.empty()) continue;

        var pathLen = path.length();
        if (pathLen <= 0) continue;

        for (var i = 0; i < count; i++) {
            var t = i / count;  // Parameter 0-1 along path
            var pos = path.pointAtParam(t);
            var normal = path.normalAtParam(t);
            var angle = path.angleAtParam(t);

            // Use pos, normal, angle to place/orient objects
        }
    }
}
```

### Resampling Paths

Use `path.resample(edgeLength)` to create evenly spaced points:

```javascript
function deformMesh(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        if (!path || path.empty()) continue;

        // Resample for even point distribution
        path.resample(resampleLength);

        var pd = path.pathData();
        // ... modify points ...
        path.setPathData(pd);
        mesh.setPathAtIndex(p, path);
    }
}
```

### Adding Geometry in Deformers

Deformers can add new paths to meshes, not just modify existing ones:

```javascript
function addGreeblesToMesh(mesh) {
    var pathCount = mesh.count();
    for (var p = 0; p < pathCount; p++) {
        var path = mesh.getPathAtIndex(p);
        if (!path || path.empty()) continue;

        for (var i = 0; i < count; i++) {
            var t = i / count;
            var pos = path.pointAtParam(t);

            // Create new geometry
            var greeblePath = new cavalry.Path();
            greeblePath.addRect(-width/2, -height/2, width/2, height/2);
            greeblePath.translate(pos.x, pos.y);

            // Add to mesh (inherits material from parent)
            mesh.addPath(greeblePath);
        }
    }
}
```

### Path Data Structure

The `path.pathData()` method returns an array of point objects:

```javascript
var pd = path.pathData();
// Each pd[idx] contains:
// - pd[idx].point.x, pd[idx].point.y  - position
// - pd[idx].normal.x, pd[idx].normal.y - outward normal
```

### Hierarchy vs Depth-Based Access

**Hierarchy:**
```javascript
var root = def.getRootMesh();
visitAllMeshes(root, myFunction);
def.setRootMesh(root);
```

**Depth-Based:**
```javascript
var depth = def.highestDepthWithPath();
var meshes = def.getMeshesAtDepth(depth);
// ... modify meshes ...
def.setMeshesAtDepth(depth, meshes);
```

The hierarchy approach handles nested groups correctly and is generally preferred.
